/**
 * \file caam_op_encrypt_decrypt.c
 *
 * \brief Architecture specific implementation of functions relevant for encrypt and decrypt
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <string.h>
#include "caam.h"
#include <sdc/arch/errors.h>

#ifdef CAAM_EE_FLAG_INTERLEAVED_OP_BIT
#ifdef CAAM_EE_FLAG_NO_CLEAR_BIT
    #define CAAM_EE_ENC_DEFAULT_FLAGS (CAAM_EE_FLAG_NO_CLEAR_BIT | CAAM_EE_FLAG_INTERLEAVED_OP_BIT)
    #define CAAM_EE_ENC_DEFAULT_FLAGS_CFB_WORKAROUND CAAM_EE_FLAG_NO_CLEAR_BIT
#else
    #define CAAM_EE_ENC_DEFAULT_FLAGS CAAM_EE_FLAG_INTERLEAVED_OP_BIT
    #define CAAM_EE_ENC_DEFAULT_FLAGS_CFB_WORKAROUND 0
#endif
#else
#ifdef CAAM_EE_FLAG_NO_CLEAR_BIT
    #define CAAM_EE_ENC_DEFAULT_FLAGS CAAM_EE_FLAG_NO_CLEAR_BIT
    #define CAAM_EE_ENC_DEFAULT_FLAGS_CFB_WORKAROUND CAAM_EE_FLAG_NO_CLEAR_BIT
#else
    #define CAAM_EE_ENC_DEFAULT_FLAGS 0
    #define CAAM_EE_ENC_DEFAULT_FLAGS_CFB_WORKAROUND 0
#endif
#endif

#define CAAM_EE_ENC_INIT_UPDATE_FINALIZE_FLAG 0  /* No interleaving mode is used */

/* define default type for encrypt/decrypt in IMX6 */
static const sdc_encrypt_decrypt_type_t sdc_encrypt_decrypt_default_type =
{SDC_ENCDEC_ALG_AES, SDC_ENCDEC_BLK_CTR, true, CAAM_EE_CIPHER_AES_FLAG, CAAM_EE_CIPHER_AES_IDX, CAAM_EE_ENCRYPT_BLOCK_CTR_FLAG, CAAM_EE_ENCRYPT_BLOCK_CTR_IDX};

/* defined in sdc_arch.h */
const sdc_encrypt_decrypt_type_t *sdc_arch_encrypt_decrypt_get_default(void)
{
    return &sdc_encrypt_decrypt_default_type;
}


/* defined in sdc_arch.h */
sdc_error_t sdc_arch_encrypt_decrypt_type_alloc(sdc_encrypt_decrypt_type_t **type)
{
    *type = malloc(sizeof(sdc_encrypt_decrypt_type_t));
    if (!(*type))
        return SDC_NO_MEM;

    /* initialize with default */
    memcpy(*type, &sdc_encrypt_decrypt_default_type, sizeof(sdc_encrypt_decrypt_type_t));

    return SDC_OK;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_encrypt_decrypt_type_free(sdc_encrypt_decrypt_type_t *type)
{
    free (type);

    return SDC_OK;
}

static sdc_error_t caam_map_sdc_ecrypt_decrypt_alg (
    sdc_encrypt_decrypt_alg_t alg,
    uint32_t *flag, uint32_t *idx,
    sdc_encrypt_decrypt_blk_t *default_blk,
    bool *is_symmetric)
{
    sdc_error_t err = SDC_OK;

    switch(alg) {
    case SDC_ENCDEC_ALG_AES:
        *flag = CAAM_EE_CIPHER_AES_FLAG;
        *idx =  CAAM_EE_CIPHER_AES_IDX;
        *default_blk = SDC_ENCDEC_BLK_CTR;
        *is_symmetric = true;
        break;
    case SDC_ENCDEC_ALG_3DES:
        *flag = CAAM_EE_CIPHER_3DES_FLAG;
        *idx =  CAAM_EE_CIPHER_3DES_IDX;
        *default_blk = SDC_ENCDEC_BLK_OFB;
        *is_symmetric = true;
        break;
    case SDC_ENCDEC_ALG_DES:
        *flag = CAAM_EE_CIPHER_DES_FLAG;
        *idx =  CAAM_EE_CIPHER_DES_IDX;
        *default_blk = SDC_ENCDEC_BLK_OFB;
        *is_symmetric = true;
        break;
    case SDC_ENCDEC_ALG_RSA:
        *flag = 0; /* unused */
        *idx = 0; /* unused */
        *default_blk = SDC_ENCDEC_BLK_NONE;
        *is_symmetric = false;
        break;
    default:
        err = SDC_ALG_MODE_INVALID;
    }

    return err;
}
static sdc_error_t caam_map_sdc_ecrypt_decrypt_symmetric_blk (sdc_encrypt_decrypt_blk_t blk, uint32_t *flag, uint32_t *idx)
{
    sdc_error_t err = SDC_OK;

    switch(blk) {
    case SDC_ENCDEC_BLK_ECB:
        *flag = CAAM_EE_ENCRYPT_BLOCK_ECB_FLAG;
        *idx =  CAAM_EE_ENCRYPT_BLOCK_ECB_IDX;
        break;
    case SDC_ENCDEC_BLK_CTR:
        *flag = CAAM_EE_ENCRYPT_BLOCK_CTR_FLAG;
        *idx =  CAAM_EE_ENCRYPT_BLOCK_CTR_IDX;
        break;
    case SDC_ENCDEC_BLK_CFB:
        *flag = CAAM_EE_ENCRYPT_BLOCK_CFB_FLAG;
        *idx =  CAAM_EE_ENCRYPT_BLOCK_CFB_IDX;
        break;
    case SDC_ENCDEC_BLK_OFB:
        *flag = CAAM_EE_ENCRYPT_BLOCK_OFB_FLAG;
        *idx =  CAAM_EE_ENCRYPT_BLOCK_OFB_IDX;
        break;
    case SDC_ENCDEC_BLK_CBC:
        *flag = CAAM_EE_ENCRYPT_BLOCK_CBC_FLAG;
        *idx =  CAAM_EE_ENCRYPT_BLOCK_CBC_IDX;
        break;
    default:
        err = SDC_ALG_MODE_INVALID;
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_encrypt_decrypt_type_set_algorithm(sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_alg_t alg)
{
    sdc_error_t err = SDC_OK;
    sdc_encrypt_decrypt_type_t tmp;

    tmp.alg = alg;
    err = caam_map_sdc_ecrypt_decrypt_alg(tmp.alg, &tmp.arch_alg_flag,
                                          &tmp.arch_alg_idx, &tmp.blk, &tmp.is_symmetric);

    if (err == SDC_OK) {
        err = sdc_encrypt_decrypt_type_set_block_mode (&tmp, tmp.blk);
    }

    if (err == SDC_OK) {
        memcpy(type, &tmp, sizeof(tmp));
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_encrypt_decrypt_type_set_block_mode(sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_blk_t blk)
{
    sdc_error_t err = SDC_OK;
    uint32_t blk_flag;
    uint32_t blk_idx;
    uint32_t alg_idx;

    if (type->is_symmetric) {
        err = caam_map_sdc_ecrypt_decrypt_symmetric_blk(blk, &blk_flag, &blk_idx);

        if (err == SDC_OK) {
            alg_idx = type->arch_alg_idx;

            /* invalid block mode */
            err = SDC_ALG_MODE_INVALID;

            /* check alg index - maybe someone has written directly to container */
            if (alg_idx < sizeof(caam_ee_cipher_descs) / sizeof(struct caam_ee_cipher_desc)) {
                /* check if block mode is supported */
                if ((caam_ee_cipher_descs[alg_idx].encrypt_blk_mode_msk & (1<<blk_idx)) != 0) {
                    type->blk = blk;
                    type->arch_blk_flag = blk_flag;
                    type->arch_blk_idx = blk_idx;
                    err = SDC_OK;
                }
            }
        }
    } else {
        if (blk != SDC_ENCDEC_BLK_NONE) {
            err = SDC_ALG_MODE_INVALID;
        } else {
            type->blk = SDC_ENCDEC_BLK_NONE;
        }
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_encrypt_decrypt_type_get_algorithm(const sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_alg_t *alg)
{
    *alg = type->alg;

    return SDC_OK;

}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_encrypt_decrypt_type_get_block_mode(const sdc_encrypt_decrypt_type_t *type, sdc_encrypt_decrypt_blk_t *blk)
{
    *blk = type->blk;

    return SDC_OK;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_encrypt_decrypt_key_desc_fill (
    const sdc_encrypt_decrypt_type_t *type,
    sdc_key_desc_t *desc)
{
    sdc_error_t err = SDC_OK;
    uint32_t alg_idx;
    uint32_t blk_idx;

    if (type->is_symmetric) {
        blk_idx = type->arch_blk_idx;
        alg_idx = type->arch_alg_idx;

        /* check range of alg */
        if (alg_idx >= sizeof(caam_ee_cipher_descs) / sizeof(struct caam_ee_cipher_desc))
            return SDC_ALG_MODE_INVALID;

        /* check if block mode is supported */
        if ((caam_ee_cipher_descs[alg_idx].encrypt_blk_mode_msk & (1<<blk_idx)) == 0)
            return SDC_ALG_MODE_INVALID;

        caam_keylen_bmsk_to_sdc(caam_ee_cipher_descs[alg_idx].key_len_msk,
                                &(desc->sup_key_lens), &(desc->dflt_key_len));
        desc->sup_key_fmt_protect   = SDC_KEY_FMT_SIMPLE_SYM;
        desc->sup_key_fmt_unprotect = SDC_KEY_FMT_SIMPLE_SYM;
    } else {
        desc->sup_key_lens = SDC_KEY_LEN_BMSK_1024bit | SDC_KEY_LEN_BMSK_2048bit | SDC_KEY_LEN_BMSK_4096bit;
        desc->dflt_key_len = SDC_KEY_LEN_4096bit;
        desc->sup_key_fmt_protect = SDC_KEY_FMT_RSA_PUBLIC;
        desc->sup_key_fmt_unprotect = SDC_KEY_FMT_RSA_PRIVATE;
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_encrypt_decrypt_desc_fill (
    sdc_session_t *session,
    const sdc_encrypt_decrypt_type_t *type,
    sdc_encrypt_decrypt_desc_t *desc)
{

    sdc_error_t err = SDC_OK;
    const struct caam_ee_blk_desc *caam_ee_encrypt_blk_desc;
    uint32_t blk_idx;
    uint32_t alg_idx;
    sdc_key_info_t key_info;

    if (type->is_symmetric) {
        /* optional check of key fmt */
        if (session) {
            err = sdc_arch_session_key_info_get(session, &key_info, false);

            if ((err == SDC_OK) &&
                (key_info.key_fmt != SDC_KEY_FMT_SIMPLE_SYM))
                err = SDC_KEY_FMT_INVALID;
        }

        blk_idx = type->arch_blk_idx;
        alg_idx = type->arch_alg_idx;

        /* check range of alg */
        if ((err == SDC_OK) && (alg_idx >= sizeof(caam_ee_cipher_descs) / sizeof(struct caam_ee_cipher_desc))) {
            err = SDC_ALG_MODE_INVALID;
        } else {
            /* check if block mode is supported */
            if ((err == SDC_OK) && ((caam_ee_cipher_descs[alg_idx].encrypt_blk_mode_msk & (1<<blk_idx)) == 0))
                err =  SDC_ALG_MODE_INVALID;

            if (err == SDC_OK) {
                caam_ee_encrypt_blk_desc = &caam_ee_cipher_descs[alg_idx].encrypt_blk_descs[blk_idx];

                desc->iv.max = caam_ee_encrypt_blk_desc->iv_max;
                desc->iv.min = caam_ee_encrypt_blk_desc->iv_min;
                desc->iv.mod = caam_ee_encrypt_blk_desc->iv_mod;
                desc->iv.dflt = desc->iv.max;

                desc->supports_iuf = true;

                desc->data.block_len = caam_ee_cipher_descs[alg_idx].block_size;
                desc->data.max_chunk_len = caam_ee_encrypt_blk_desc->max_block_len;
                desc->data.chunk_len_aligned = true;

                desc->data.padding = SDC_PADDING_NO;
                if (caam_ee_encrypt_blk_desc->final_block_aligned) {
                    if (type->blk == SDC_ENCDEC_BLK_CFB) {
                        /* imx6 requires block aligned buffers for CFB last block - correct in userspace */
                        desc->data.padding = SDC_PADDING_NO;
                    } else {
                        desc->data.padding = SDC_PADDING_PKCS7;
                    }
                }

                /* calculate max and alignment for plain and cipher */
                err = sdc_intern_get_plain_cipher_spec(
                    desc->data.padding,
                    0,
                    caam_ee_encrypt_blk_desc->max_total_len,
                    desc->data.block_len,
                    &(desc->data.plain),
                    &(desc->data.cipher));
            }
        }
    } else {
        /* session is mandatory - we need to read the key length */
        if (!session)
            err = SDC_SESSION_INVALID;

        if (err == SDC_OK) {
            err = sdc_arch_session_key_info_get(session, &key_info, false);

            if ((err == SDC_OK) &&
                (key_info.key_fmt != SDC_KEY_FMT_RSA_PRIVATE) &&
                (key_info.key_fmt != SDC_KEY_FMT_RSA_PUBLIC))
                err = SDC_KEY_FMT_INVALID;

            if (err == SDC_OK) {
                /* for asym. crypto we can't use init update finalize API */
                desc->supports_iuf = false;

                desc->data.block_len = key_info.key_len_bytes;;
                desc->data.max_chunk_len = key_info.key_len_bytes;;
                desc->data.chunk_len_aligned = true;
                desc->data.padding = SDC_PADDING_INTERNAL;

                desc->data.cipher.min = key_info.key_len_bytes;
                desc->data.cipher.max = key_info.key_len_bytes;
                desc->data.cipher.aligned = false;

                desc->data.plain.min = 0;
                desc->data.plain.max = CAAM_EE_RSA_ENC_PKCS1_V15_MAX_PLAIN_LEN(key_info.key_len_bytes);
                desc->data.plain.aligned = false;

                desc->iv.min = 0;
                desc->iv.max = 0;
                desc->iv.mod = 0;
                desc->iv.dflt = 0;
            }
        }
    }

    return err;
}

static sdc_error_t caam_asymmetric_encrypt_decrypt(sdc_session_t *session,
                                                   caam_encrypt_decrypt_op operation,
                                                   const sdc_encrypt_decrypt_type_t *type,
                                                   const sdc_encrypt_decrypt_desc_t *desc,
                                                   const uint8_t *in_data,
                                                   const size_t in_data_len,
                                                   uint8_t *out_data,
                                                   size_t *out_data_len)
{
    sdc_error_t err = SDC_OK;
    int res;
    int res_errno;
    unsigned long int request;
    struct caam_ee_asym_encrypt_decrypt_params iodata;

    /* we only support RSA PKCS1_V1.5 */
    (void)session;
    (void)type;
    (void)desc;
    iodata.in = (uint8_t*)in_data;
    iodata.in_len = in_data_len;
    iodata.out = out_data;
    iodata.out_len = *out_data_len;
    iodata.flags = CAAM_EE_ASYM_ENC_DEC_PAD_PKCS1_V15;

    if (operation == ENCRYPT) {
        request = CAAM_ASYM_ENCRYPT_DATA;
    } else {
        request = CAAM_ASYM_DECRYPT_DATA;
    }

    res = ioctl(session->arch.fd, request, &iodata);
    if (res != 0) {
        res_errno = errno;
        err = error_from_ioctl_errno(res_errno, request, iodata.flags);
    }

    *out_data_len = iodata.out_len;

    return err;
}

static sdc_error_t caam_symmetric_encrypt_decrypt(sdc_session_t *session,
                                                  caam_encrypt_decrypt_op operation,
                                                  const sdc_encrypt_decrypt_type_t *type,
                                                  const sdc_encrypt_decrypt_desc_t *desc,
                                                  const uint8_t *in_data,
                                                  const size_t in_data_len,
                                                  uint8_t *out_data,
                                                  const size_t out_data_len,
                                                  const uint8_t *iv, const size_t iv_len)
{
    sdc_error_t err = SDC_OK;
    uint8_t padding_buffer[CAAM_EE_MAX_ENC_BLOCK_ALIGN];
    const uint8_t *in_data_ptr;
    uint8_t *out_data_ptr;
    size_t in_len_remaining;
    size_t out_len_remaining;
    size_t buf_idx;
    size_t buf_len;
    size_t align;
    size_t stop_len;
    struct caam_ee_encrypt_decrypt_params iodata;
    int res;
    int res_errno;
    unsigned long int request;
    bool padding_ioctl;
    uint8_t imx6_cfb_buffer[CAAM_EE_MAX_ENC_BLOCK_ALIGN];
    bool imx6_cfb_workaround;

    if (desc->data.block_len > CAAM_EE_MAX_ENC_BLOCK_ALIGN) {
        /* buffer not sufficient to handle padding */
        return SDC_UNKNOWN_ERROR;
    }

    if (caam_is_op_in_progress(session)) {
        /* Cancel non-completed operation if any */
        err = caam_perform_abort_operation(session);
        if (err != SDC_OK)
            return err;
    }

    in_data_ptr = in_data;
    out_data_ptr = out_data;
    in_len_remaining = in_data_len;
    out_len_remaining = out_data_len;
    buf_idx = 0;

    memset(&imx6_cfb_buffer[0], 0, sizeof(uint8_t)*CAAM_EE_MAX_ENC_BLOCK_ALIGN);

    iodata.iv = (uint8_t*)iv;
    iodata.iv_len = (size_t)iv_len;

    /* Initialize to no padding required
     * If no padding is required the complete data is handled by the
     * while loop only */
    stop_len = 0;

    if (operation == ENCRYPT) {
        request = CAAM_ENCRYPT_DATA;
    } else {
        request = CAAM_DECRYPT_DATA;
    }

    /* On IMX6 CFB block mode is limited to aligned last buffers
     * In order to overcome this limitation a workaround in userspace is used
     * This workaround only needs to be used if length is not already aligned
     *
     * If required this workaround will always use padding buffers for input
     * and output
     * In order to do so interleaved mode will be disabled */
    imx6_cfb_workaround = false;
    if ((in_data_len % desc->data.block_len) && (type->blk == SDC_ENCDEC_BLK_CFB)) {
        imx6_cfb_workaround = true;
        stop_len = desc->data.block_len - 1;
        memset(padding_buffer, 0, CAAM_EE_MAX_ENC_BLOCK_ALIGN);

        /* cfb requires equal input and output length */
        if (in_data_len != out_data_len) {
            return SDC_INTERNAL_ERROR;
        }
    } else {
        if (operation == ENCRYPT) {
            if (desc->data.padding != SDC_PADDING_NO) {
                /* If padding is required data is always processed by the while loop
                 * and the padding logic.
                 * In the in-data is aligned or 0 all data will be handled by the
                 * while loop completely and the padding logic will add a complete
                 * block of padding.
                 * In other cases up to stop_len == block-alignment-1 bytes are not
                 * handled by the while loop but by the padding logic */
                stop_len = desc->data.block_len - 1;
            } else {
                /* without padding both lengths need to be equal */
                if (in_data_len != out_data_len) {
                    return SDC_INTERNAL_ERROR;
                }
            }
        } else {
            /* input and output length need to be equal */
            if (in_data_len != out_data_len) {
                return SDC_INTERNAL_ERROR;
            }
        }
    }

    if (imx6_cfb_workaround) {
        iodata.flags = type->arch_alg_flag | type->arch_blk_flag | CAAM_EE_ENC_DEFAULT_FLAGS_CFB_WORKAROUND;
    } else {
        iodata.flags = type->arch_alg_flag | type->arch_blk_flag | CAAM_EE_ENC_DEFAULT_FLAGS;
    }

    while ((in_len_remaining > stop_len) && (err == SDC_OK)) {
        buf_len = in_len_remaining;

        if (buf_len > desc->data.max_chunk_len)
            buf_len = desc->data.max_chunk_len;

        in_len_remaining -= buf_len;

        iodata.flags &= ~CAAM_OP_MASK;

        if ((in_len_remaining == 0) && (stop_len == 0)) {
            /* SINGLE or MULTI_FINALIZE will only happen if no padding is required */

            /* final or init-final */
            if (buf_idx == 0) {
                iodata.flags |= CAAM_OP_SINGLE;
            } else {
                iodata.flags |= CAAM_OP_MULTI_FINALIZE;
            }
        } else {
            /* init or update */

            /* alignment */
            align = buf_len % desc->data.block_len;
            buf_len -= align;
            in_len_remaining += align;

            if (buf_idx == 0) {
                iodata.flags |= CAAM_OP_MULTI_INIT;
            } else {
                iodata.flags |= CAAM_OP_MULTI_UPDATE;
            }
        }

        iodata.in = (uint8_t*)in_data_ptr;
        iodata.out = out_data_ptr;
        iodata.in_len = buf_len;
        iodata.out_len = out_len_remaining;

        res = ioctl(session->arch.fd, request, &iodata);

        if (res == 0) {
            buf_idx++;
            in_data_ptr += buf_len;
            out_data_ptr += iodata.out_len;
            out_len_remaining -= iodata.out_len;
        } else {
            res_errno = errno;

            /* ignore and retry after EAGAIN */
            if (res_errno != EAGAIN) {
                err = error_from_ioctl_errno(res_errno, request, iodata.flags);
            }

            in_len_remaining += buf_len;
        }
    }

    /*
     * in case we need padding or cfb_workaround (stop_len > 0) or
     * no request so far (buf_idx == 0) - even in case data_len is 0 we need to send one ioctl (checking key permissions)
     * */
    if (((stop_len) || (buf_idx == 0)) && (err == SDC_OK)) {

        iodata.flags &= ~CAAM_OP_MASK;
        if (buf_idx == 0) {
            iodata.flags |= CAAM_OP_SINGLE;
        } else {
            iodata.flags |= CAAM_OP_MULTI_FINALIZE;
        }

        /* stop_len > 0 <==> padding required (data length == block length) */
        if (stop_len) {
            /* 0 <= in_len_remaining <= stop_len = block-alignment - 1 */
            if (in_len_remaining) {
                /* copy remaining_data to padding buffer */
                memcpy(padding_buffer, in_data_ptr, in_len_remaining);
            }

            /* CFB workaround won't use normal padding */
            if (!imx6_cfb_workaround) {
                err = sdc_intern_pad(desc->data.padding, padding_buffer, in_len_remaining, desc->data.block_len);
            }

            iodata.in = padding_buffer;
            iodata.in_len = desc->data.block_len;
        } else {
            /*
             * In this case no ioctl was send so far (buf_idx == 0), but no padding or cfb_workaround required.
             * The previous while loop was never executed as (buf_idx == 0).
             * Therefore in_len_remaining needs to be 0.
             * Output length will be 0 as well.
             * The IOCTL is only send to enforce checking of key permissions.
             */
            iodata.in = NULL;
            iodata.in_len = 0;
        }

        if (imx6_cfb_workaround) {
            /* use a special buffer for the workaround - output is to small */
            iodata.out = imx6_cfb_buffer;
            iodata.out_len = desc->data.block_len;
        } else {
            iodata.out = out_data_ptr;
            iodata.out_len = out_len_remaining;
        }

        padding_ioctl = true;
        while ((padding_ioctl) && (err == SDC_OK)) {
            res = ioctl(session->arch.fd, request, &iodata);
            if (res != 0) {
                res_errno = errno;

                /* ignore and retry after EAGAIN */
                if (res_errno != EAGAIN) {
                    err = error_from_ioctl_errno(res_errno, request, iodata.flags);
                }
            } else {
                padding_ioctl = false;
            }
        }

        if ((err == SDC_OK) && (imx6_cfb_workaround)) {
            /* copy relevant cfb data from workaround buffer to output */
            memcpy(out_data_ptr, imx6_cfb_buffer, out_len_remaining);
        }
    }

    sdc_intern_overwrite_secret(imx6_cfb_buffer, CAAM_EE_MAX_ENC_BLOCK_ALIGN);
    sdc_intern_overwrite_secret(padding_buffer, CAAM_EE_MAX_ENC_BLOCK_ALIGN);

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_encrypt_get_out_len(sdc_session_t *session,
                                         const sdc_encrypt_decrypt_type_t *type,
                                         const sdc_encrypt_decrypt_desc_t *desc,
                                         const size_t in_len,
                                         size_t *out_len)
{
    sdc_error_t err = SDC_OK;

    (void)session;

    if (type->is_symmetric) {
        err = sdc_intern_get_cipher_len(
            desc->data.padding,
            in_len,
            desc->data.block_len,
            out_len);
    } else {
        if (desc->data.padding != SDC_PADDING_INTERNAL)
            err = SDC_INTERNAL_ERROR;

        if ((err == SDC_OK) && (in_len > desc->data.plain.max))
            err = SDC_IN_DATA_INVALID;

        if (err == SDC_OK)
            *out_len = desc->data.cipher.max;
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_encrypt(sdc_session_t *session,
                             const sdc_encrypt_decrypt_type_t *type,
                             const sdc_encrypt_decrypt_desc_t *desc,
                             const uint8_t *in_data,
                             const size_t in_data_len,
                             uint8_t *out_data,
                             const size_t out_data_len,
                             const uint8_t *iv, const size_t iv_len)
{
    size_t out_tmp;

    if (type->is_symmetric) {
        return caam_symmetric_encrypt_decrypt (session, ENCRYPT, type, desc, in_data,
                                               in_data_len, out_data, out_data_len, iv, iv_len);
    }

    out_tmp = out_data_len;

    return caam_asymmetric_encrypt_decrypt(session, ENCRYPT, type, desc, in_data,
                                           in_data_len, out_data, &out_tmp);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_decrypt_get_max_out_len(sdc_session_t *session,
                                             const sdc_encrypt_decrypt_type_t *type,
                                             const sdc_encrypt_decrypt_desc_t *desc,
                                             const size_t in_len,
                                             size_t *out_max_len)
{
    sdc_error_t err = SDC_OK;

    if (type->is_symmetric) {
        err = sdc_intern_get_max_plain_len(
            desc->data.padding,
            in_len,
            desc->data.block_len,
            out_max_len);
    } else {
        if (desc->data.padding != SDC_PADDING_INTERNAL)
            err = SDC_INTERNAL_ERROR;

        /* session is mandatory - we need to read the key length */
        if ((err == SDC_OK) && (!session))
            err = SDC_SESSION_INVALID;

        /* cipher input needs to be equal to key length */
        if ((err == SDC_OK) && (in_len != desc->data.cipher.max))
            err = SDC_IN_DATA_INVALID;

        if (err == SDC_OK)
            *out_max_len = desc->data.plain.max;
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_decrypt(sdc_session_t *session,
                             const sdc_encrypt_decrypt_type_t *type,
                             const sdc_encrypt_decrypt_desc_t *desc,
                             const uint8_t *in_data,
                             const size_t in_data_len,
                             uint8_t *out_data,
                             size_t *out_data_len,
                             const uint8_t *iv, const size_t iv_len)
{
    sdc_error_t err;

    if (type->is_symmetric) {
        err = caam_symmetric_encrypt_decrypt (session, DECRYPT, type, desc, in_data,
                                              in_data_len, out_data, *out_data_len, iv, iv_len);


        if (err == SDC_OK) {
            err = sdc_intern_unpad(desc->data.padding, out_data, out_data_len, desc->data.block_len);
        }
    } else {
        err = caam_asymmetric_encrypt_decrypt(session, DECRYPT, type, desc, in_data,
                                              in_data_len, out_data, out_data_len);
    }

    return err;
}

sdc_error_t sdc_arch_encrypt_get_update_len(sdc_session_t *session,
                                            const sdc_encrypt_decrypt_type_t *type,
                                            const sdc_encrypt_decrypt_desc_t *desc,
                                            const size_t in_len,
                                            size_t *out_len_max)
{
    (void)session;
    (void)type;
    (void)desc;
    /* caam will always produce the output length == input length */
    *out_len_max = in_len;

    return SDC_OK;
}

sdc_error_t sdc_arch_decrypt_get_update_len(sdc_session_t *session,
                                            const sdc_encrypt_decrypt_type_t *type,
                                            const sdc_encrypt_decrypt_desc_t *desc,
                                            const size_t in_len,
                                            size_t *out_len_max)
{
    (void)session;
    (void)type;
    (void)desc;
    /* caam will always produce the output length == input length */
    *out_len_max = in_len;

    return SDC_OK;
}

static sdc_error_t caam_arch_encrypt_decrypt_init(sdc_session_t *session,
                                                  const sdc_encrypt_decrypt_desc_t *desc,
                                                  const uint8_t *iv, size_t iv_len,
                                                  int operation)
{

    sdc_error_t err = SDC_OK;

    if (desc->data.block_len > CAAM_EE_MAX_ENC_BLOCK_ALIGN) {
        /* buffer not sufficient to handle padding */
        return SDC_UNKNOWN_ERROR;
    }

    if (caam_is_op_in_progress(session)) {
        /* Cancel non-completed operation if any */
        err = caam_perform_abort_operation(session);
        if (err != SDC_OK)
            return err;
    }

    memcpy(session->arch.iv, (uint8_t *)iv, iv_len);
    session->arch.iv_len = (size_t) iv_len;

    if (operation == ENCRYPT) {
        session->arch.request = CAAM_ENCRYPT_DATA;
    } else {
        session->arch.request = CAAM_DECRYPT_DATA;
    }

    session->arch.buf_idx = 0;

    return err;

}

static sdc_error_t caam_encrypt_decrypt_update(sdc_session_t *session,
                                               const sdc_encrypt_decrypt_desc_t *desc,
                                               const sdc_encrypt_decrypt_type_t *type,
                                               const uint8_t *in_data, const size_t in_data_len,
                                               uint8_t *out_data, size_t *out_data_len )
{
    sdc_error_t err = SDC_OK;
    struct caam_ee_encrypt_decrypt_params iodata;
    uint32_t iodata_flags = type->arch_alg_flag | type->arch_blk_flag | CAAM_EE_ENC_INIT_UPDATE_FINALIZE_FLAG;
    int res_errno = 0;
    int res;

    iodata.iv = (uint8_t*) session->arch.iv;
    iodata.iv_len = (size_t) session->arch.iv_len;

    /* output needs to be always greater or equal to input */
    if (in_data_len > *out_data_len)
        err = SDC_OUT_DATA_INVALID;

    /* data needs to be aligned */
    if ((in_data_len % desc->data.block_len) != 0)
        err = SDC_INTERNAL_ERROR;

    if (err == SDC_OK) {
        /* init or update */
        if (session->arch.buf_idx == 0) {
            iodata_flags |= CAAM_OP_MULTI_INIT;
        } else {
            iodata_flags |= CAAM_OP_MULTI_UPDATE;
        }

        /* retry in case of EAGAIN */
        while (err == SDC_OK) {
            res = caam_perform_encrypt_decrypt_ioctl_operation(session, &iodata,
                                                               in_data,
                                                               out_data,
                                                               in_data_len,
                                                               *out_data_len,
                                                               iodata_flags);
            res_errno = errno;

            if (res == 0) {
                break;
            } else {
                /* ignore and retry after EAGAIN */
                if (res_errno != EAGAIN) {
                    err = error_from_ioctl_errno(res_errno, session->arch.request, iodata_flags);
                }
            }
        }
        if (err == SDC_OK) {
            session->arch.buf_idx++;
            *out_data_len = iodata.out_len;
        }
    }

    return err;
}

static sdc_error_t caam_encrypt_decrypt_finalize(sdc_session_t *session,
                                                 const sdc_encrypt_decrypt_type_t *type,
                                                 const sdc_encrypt_decrypt_desc_t *desc,
                                                 const uint8_t *in_data, const size_t in_data_len,
                                                 uint8_t *out_data, size_t *out_data_len)
{
    sdc_error_t err = SDC_OK;
    struct caam_ee_encrypt_decrypt_params iodata;
    uint32_t iodata_flags = type->arch_alg_flag | type->arch_blk_flag | CAAM_EE_ENC_INIT_UPDATE_FINALIZE_FLAG;
    int res_errno = 0;
    int res;
    bool imx6_cfb_workaround;
    uint8_t imx6_cfb_buffer_in[CAAM_EE_MAX_ENC_BLOCK_ALIGN];
    uint8_t imx6_cfb_buffer_out[CAAM_EE_MAX_ENC_BLOCK_ALIGN];
    size_t block_len = desc->data.block_len;
    const uint8_t *io_in_ptr = in_data;
    uint8_t *io_out_ptr = out_data;
    size_t io_in_len = in_data_len;
    size_t io_out_len = *out_data_len;

     /* in data must not exceed one block */
    if (in_data_len > block_len)
        err = SDC_INTERNAL_ERROR;

    iodata.iv = (uint8_t*) session->arch.iv;
    iodata.iv_len = (size_t) session->arch.iv_len;

    /* output needs to be always greater or equal to input */
    if (io_in_len > io_out_len)
        err = SDC_OUT_DATA_INVALID;

    imx6_cfb_workaround = false;
    if ((err == SDC_OK) &&
        ((in_data_len % desc->data.block_len) != 0) &&
        (type->blk == SDC_ENCDEC_BLK_CFB))
    {
        imx6_cfb_workaround = true;

        memset(imx6_cfb_buffer_in, 0, block_len);
        memcpy(imx6_cfb_buffer_in, io_in_ptr, in_data_len);

        io_in_ptr = imx6_cfb_buffer_in;
        io_in_len = block_len;
        io_out_ptr = imx6_cfb_buffer_out;
        io_out_len = block_len;
    }

    if (err == SDC_OK) {
        /* init or update */
        if (session->arch.buf_idx == 0) {
            iodata_flags |= CAAM_OP_SINGLE;
        } else {
            iodata_flags |= CAAM_OP_MULTI_FINALIZE;
        }

        /* retry in case of EAGAIN */
        while (err == SDC_OK) {
            res = caam_perform_encrypt_decrypt_ioctl_operation(session, &iodata,
                                                               io_in_ptr,
                                                               io_out_ptr,
                                                               io_in_len,
                                                               io_out_len,
                                                               iodata_flags);
            res_errno = errno;

            if (res == 0) {
                break;
            } else {
                /* ignore and retry after EAGAIN */
                if (res_errno != EAGAIN) {
                    err = error_from_ioctl_errno(res_errno, session->arch.request, iodata_flags);
                }
            }
        }

        if (err == SDC_OK) {
            io_out_len = iodata.out_len;

            if (imx6_cfb_workaround)
            {
                /* we are interested in the beginning of  the buffer */
                memcpy(out_data, io_out_ptr, in_data_len);
                *out_data_len = in_data_len;
            } else {
                *out_data_len = io_out_len;
            }
        }
    }

    if (imx6_cfb_workaround) {
        /* make sure the buffer doesn't contain any confidential data */
        sdc_intern_overwrite_secret(imx6_cfb_buffer_in, CAAM_EE_MAX_ENC_BLOCK_ALIGN);
        sdc_intern_overwrite_secret(imx6_cfb_buffer_out, CAAM_EE_MAX_ENC_BLOCK_ALIGN);
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_encrypt_init(sdc_session_t *session,
                                  const sdc_encrypt_decrypt_type_t *type,
                                  const sdc_encrypt_decrypt_desc_t *desc,
                                  const uint8_t *iv, size_t iv_len)
{
    if(type->is_symmetric != true)
        return SDC_NOT_SUPPORTED;

    return caam_arch_encrypt_decrypt_init(session, desc, iv, iv_len, ENCRYPT);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_decrypt_init(sdc_session_t *session,
                                  const sdc_encrypt_decrypt_type_t *type,
                                  const sdc_encrypt_decrypt_desc_t *desc,
                                  const uint8_t *iv, size_t iv_len)
{
    if(type->is_symmetric != true)
        return SDC_NOT_SUPPORTED;

    return caam_arch_encrypt_decrypt_init(session, desc, iv, iv_len, DECRYPT);

}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_encrypt_update(sdc_session_t *session,
                                    const sdc_encrypt_decrypt_type_t *type,
                                    const sdc_encrypt_decrypt_desc_t *desc,
                                    const uint8_t *in_data, const size_t in_data_len,
                                    uint8_t *out_data, size_t *out_data_len)
{
    return caam_encrypt_decrypt_update(session, desc, type, in_data,
                                       in_data_len, out_data, out_data_len);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_decrypt_update(sdc_session_t *session,
                                    const sdc_encrypt_decrypt_type_t *type,
                                    const sdc_encrypt_decrypt_desc_t *desc,
                                    const uint8_t *in_data, const size_t in_data_len,
                                    uint8_t *out_data, size_t *out_data_len)
{
    return caam_encrypt_decrypt_update(session, desc, type, in_data,
                                       in_data_len, out_data, out_data_len);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_encrypt_finalize(sdc_session_t *session,
                                      const sdc_encrypt_decrypt_type_t *type,
                                      const sdc_encrypt_decrypt_desc_t *desc,
                                      const uint8_t *in_data, const size_t in_data_len,
                                      uint8_t *out_data, size_t *out_data_len)
{
    return caam_encrypt_decrypt_finalize(session, type, desc,
                                         in_data, in_data_len,
                                         out_data, out_data_len);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_decrypt_finalize(sdc_session_t *session,
                                      const sdc_encrypt_decrypt_type_t *type,
                                      const sdc_encrypt_decrypt_desc_t *desc,
                                      const uint8_t *in_data, const size_t in_data_len,
                                      uint8_t *out_data, size_t *out_data_len)
{
    return caam_encrypt_decrypt_finalize(session, type, desc,
                                         in_data, in_data_len,
                                         out_data, out_data_len);
}
